<?php declare(strict_types=1);

namespace Nadybot\User\Modules\WIKI_MODULE;

use Amp\{CancelledException, TimeoutCancellation};
use Amp\Http\Client\{HttpClientBuilder, Request, TimeoutException};
use Nadybot\Core\{
	Attributes as NCA,
	CmdContext,
	Hydrator,
	ModuleInstance,
	Safe,
	Text,
};
use Nadybot\Core\Exceptions\UserException;
use Nadybot\Core\Types\AccessLevel;
use Nadylib\Type;
use Psr\Log\LoggerInterface;
use Safe\Exceptions\JsonException;
use Throwable;

/**
 * @author Nadyita (RK5) <nadyita@hodorraid.org>
 */

#[
	NCA\Instance,
	NCA\DefineCommand(
		command: 'wiki',
		accessLevel: AccessLevel::Guest,
		description: 'Look up a word in Wikipedia',
	)
]
class WikiController extends ModuleInstance {
	#[NCA\Inject]
	private HttpClientBuilder $http;

	#[NCA\Logger]
	private LoggerInterface $logger;

	/** Look up Wikipedia entries */
	#[NCA\HandlesCommand('wiki')]
	public function wikiCommand(CmdContext $context, string $search): void {
		$client = $this->http->build();
		$request = new Request('https://en.wikipedia.org/w/api.php?'.http_build_query([
			'format' => 'json',
			'action' => 'query',
			'prop' => 'extracts',
			'exintro' => 1,
			'explaintext' => 1,
			'redirects' => 1,
			'titles' => html_entity_decode($search),
		]));
		try {
			$response = $client->request($request, new TimeoutCancellation(10));
			if ($response->getStatus() !== 200) {
				$context->reply('Error retrieving data from Wikipedia: '. $response->getReason());
				return;
			}
			$body = $response->getBody()->buffer();
			$blob = $this->handleExtractResponse($body);
			$context->reply($blob);
		} catch (CancelledException | TimeoutException $e) {
			$context->reply("Wikipedia-request for <highlight>{$search}<end> timed out.");
			return;
		} catch (UserException $e) {
			$context->reply($e->getMessage());
			return;
		} catch (Throwable $e) {
			$context->reply('Error retrieving data from Wikipedia: '. $e->getMessage());
			return;
		}
	}

	/** Handle the response for a list of links origination from a page */
	public function handleLinksResponse(string $body): string {
		$linkPage = $this->parseResponseIntoWikiPage($body);
		$blobs = array_map(
			static function (WikiLink $link): string {
				return Text::makeChatCmd($link->title, '/tell <myname> wiki ' . $link->title);
			},
			$linkPage->links
		);
		$blob = implode("\n", $blobs);
		return Text::makeBlob($linkPage->title . ' (disambiguation)', $blob);
	}

	/** Handle the response for a wiki page */
	public function handleExtractResponse(string $body): string {
		$page = $this->parseResponseIntoWikiPage($body);
		$client = $this->http->build();

		// In case we have a page that gives us a list of terms, but no exact match,
		// query for all links in that page and present them
		if (Safe::pregMatches('/may refer to:$/m', $page->extract??'')) {
			$request = new Request('https://en.wikipedia.org/w/api.php?'.http_build_query([
				'format' => 'json',
				'action' => 'query',
				'prop' => 'links',
				'pllimit' => 'max',
				'redirects' => 1,
				'plnamespace' => 0,
				'titles' => $page->title,
			]));
			$response = $client->request($request, new TimeoutCancellation(10));
			if ($response->getStatus() !== 200) {
				throw new UserException('Error retrieving data from Wikipedia: '. $response->getReason());
			}
			$body = $response->getBody()->buffer();
			return $this->handleLinksResponse($body);
		}
		$request = new Request('https://en.wikipedia.org/w/api.php?'.http_build_query([
			'format' => 'json',
			'action' => 'parse',
			'prop' => 'text',
			'pageid' => $page->pageid,
		]));
		$response = $client->request($request, new TimeoutCancellation(10));
		if ($response->getStatus() !== 200) {
			throw new UserException('Error retrieving data from Wikipedia: '. $response->getReason());
		}
		$body = $response->getBody()->buffer();
		return $this->handleParseResponse($body, $page);
	}

	/** Handle the response for a list of links origination from a page */
	public function handleParseResponse(string $body, WikiPage $page): string {
		try {
			$wikiData = Safe::jsonDecode(
				$body,
				Type\shape([
					'parse' => Type\shape([
						'text' => Type\shape([
							'*' => Type\string(),
						], true),
					], true),
				], true),
				8
			);
		} catch (JsonException $e) {
			throw new UserException("Unable to parse Wikipedia's reply.", previous: $e);
		}
		$blob = $page->extract??'';
		$blob = Safe::pregReplace('/([a-z0-9])\.([A-Z])/', '$1. $2', $blob);
		$links = [];
		$matches = Safe::pregMatchAll(
			"/(.)<a href=\"\/wiki\/(.+?)\".*?>(.*?)<\/a>(.)/",
			$wikiData['parse']['text']['*']
		);
		for ($i = 0; $i < count($matches[1]); $i++) {
			if (!strlen($matches[3][$i]) || !strlen($matches[2][$i])) {
				continue;
			}
			$links[$matches[1][$i].$matches[3][$i].$matches[4][$i]] = urldecode(str_replace('_', ' ', $matches[2][$i]));
		}
		uksort(
			$links,
			static function (string $key1, string $key2): int {
				return strlen($key2) <=> strlen($key1);
			}
		);
		foreach ($links as $text => $link) {
			$blob = Safe::pregReplaceCallback(
				'/' . preg_quote($text, '/') . '/',
				/** @param array{0:array{string,int}} $matches */
				static function (array $matches) use ($blob, $text, $link): string {
					/** @disregard P1006 */
					$leftOpen = strrpos(substr($blob, 0, $matches[0][1]), '<a');

					/** @disregard P1006 */
					$leftClose = strrpos(substr($blob, 0, $matches[0][1]), '</a');
					if ($leftOpen > $leftClose) {
						return $matches[0][0];
					}
					return substr($text, 0, 1).
						Text::makeChatCmd(substr($text, 1, -1), "/tell <myname> wiki {$link}").
						substr($text, -1);
				},
				$blob,
				1,
				$count,
				\PREG_OFFSET_CAPTURE
			);
		}
		return Text::makeBlob($page->title, $blob);
	}

	/**
	 * Parse the AsyncHttp reply into a WikiPage object or null on error
	 *
	 * @throws UserException on invalid data
	 */
	protected function parseResponseIntoWikiPage(string $body): WikiPage {
		try {
			$wikiData = Safe::jsonDecode(
				$body,
				Type\shape([
					'query' => Type\shape([
						'pages' => Type\mixedDict(),
					], true),
				], true),
				8
			);
		} catch (JsonException $e) {
			$this->logger->error('Unable to parse Wikipedia\'s reply: {error}', [
				'body' => $body,
				'error' => $e->getMessage(),
				'exception' => $e,
			]);
			throw new UserException("Unable to parse Wikipedia's reply.", previous: $e);
		}

		/** @var list<int|string> */
		$pages = array_keys($wikiData['query']['pages']);
		$pageID = array_shift($pages);
		if ($pageID === null) {
			throw new UserException("Couldn't find a Wikipedia entry");
		}
		$page = $wikiData['query']['pages'][$pageID];
		if ($pageID < 0) {
			throw new UserException("Couldn't find a Wikipedia entry for <highlight>{$page['title']}<end>.");
		}
		$page['pageid'] = (int)$pageID;
		return Hydrator::hydrate(WikiPage::class, $page);
	}
}
